/***************************************************************************
 *   Copyright (C) 2005 by Andreas Pokorny                                 *
 *   andreas.pokorny@biozentrum.uni-wuerzburg.de                           *
 *                                                                         *
 *   This file is part of profdist and cbcanalyzer                         *
 *                                                                         *
 *   Both profdist and cbcanalyzer are free software; you can redistribute * 
 *   it and/or modify it under the terms of the GNU General Public License * 
 *   as published by the Free Software Foundation; either version 2 of the * 
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   Profdist and cbcanalyzer are distributed in the hope that it will be  *
 *   useful, but WITHOUT ANY WARRANTY; without even the implied warranty   *
 *   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef PROFDIST_PROFILE_H_INCLUDED
#define PROFDIST_PROFILE_H_INCLUDED

#include <vector>
#include <iostream>
#include "types.h"
#include "tree.h"
#include "parsed_sequence.h"
#include "aligncode.h"

/**
 * The Profile data structure. 
 * 
 * coloumn-row order?
 * ..
 *
 * struct single_profile 
 *   size_t num sequences?
 *   vector profiles 
 */
class Profile {
  public:
    typedef double value_type;
    typedef std::vector<value_type> con_type; 
    typedef con_type::const_iterator const_iterator;
    typedef con_type::iterator iterator;

    
   
    class ConstSingleProfile;

    /**
     * \brief light weight wrapper class to improve profile access
     * SingleProfile only stores the offset in the big profile array. 
     * It allows non-const access onto the data in the profile, so beware!
     */
    class SingleProfile {
      public:
        friend class ConstSingleProfile;
      private: 
        value_type* data;
        std::size_t num_rows;
        std::size_t num_sites;
      public:
        SingleProfile( value_type* d, std::size_t num_rows, std::size_t num_sites );
        /**
         * @brief Allows direct access to a column in the profile structure
         */
        inline value_type const* get_column( std::size_t site_index ) const
        { return data  + site_index * num_rows; }
        inline value_type * get_column( std::size_t site_index )
        { return data  + site_index * num_rows; }

        inline value_type const& get_value( std::size_t site_index, char row ) const
        { return *(get_column( site_index ) + row); }
        inline value_type & get_value( std::size_t site_index, char row )
        { return *(get_column( site_index ) + row); }

        void normalize( std::size_t num_sequences_in_profile );

        SingleProfile& operator=( ConstSingleProfile const& right );
        SingleProfile& operator+=( ConstSingleProfile const& right );

    };
    /**
     * \brief light weight wrapper class like SingleProfile to improve read-only profile access
     * ConstSingleProfile only stores 
     */
    class ConstSingleProfile {
      private:
        value_type const* data;
        std::size_t num_rows, num_sites;
      public:
        friend class SingleProfile;

        ConstSingleProfile( SingleProfile const& ref );
        ConstSingleProfile( value_type const* d, std::size_t num_rows, std::size_t num_sites );
        /**
         * @brief Allows direct access to a column in the profile structure
         */
        inline value_type const* get_column( std::size_t site_index ) const
        { return data  + site_index * num_rows; }

        inline value_type const& get_value( std::size_t site_index, char row ) const
        { return *(get_column( site_index ) + row); }
    };


    Profile( std::size_t num_sites, std::size_t num_prof, std::size_t num_rows = 5 );
    Profile( AlignCode const& source, tree_types::profile_map const& tree, std::size_t num_rows = 5 );
    Profile( AlignCode const& source, std::list<std::list<std::size_t> > const& profiles, std::size_t num_rows = 5 );

    template<typename IteratorT>
    Profile( std::list<sequence<IteratorT> > const& seq );


    void init( std::size_t num_sites, std::size_t num_prof, std::size_t rows = 5 );
    void init( AlignCode const& source, tree_types::profile_map const& tree, std::size_t rows = 5 );
    void init( AlignCode const& source, std::list<std::list<std::size_t> > const& profiles, std::size_t num_rows = 5 );

    /**
     * \brief Refines the profiles stored in this container.
     * This method fastly creates new profiles from the already stored profiles, by combineing the 
     * current profiles stored in this object, as stated by the list of profiles given by new_profiles
     * \param[in] new_profiles a list of new profiles used to refine this container
     */
    void refine( tree_types::profile_map const& new_profiles );
    /**
     * @brief Allows direct access to a profile in the profile structure
     */
    SingleProfile get_profile( std::size_t profile_index );
    
    /**
     * @brief Allows direct access to a profile in the profile structure
     */
    SingleProfile operator[]( std::size_t site_index );

    /**
     * @brief Allows direct read-only access to a profile in the profile structure
     */
    ConstSingleProfile get_profile( std::size_t profile_index ) const;
    
    /**
     * @brief Allows direct read-only access to a profile in the profile structure
     */
    ConstSingleProfile operator[]( std::size_t site_index ) const;

    
    /**
     * @brief returns the number of elements (sites, or columns)
     * in this profile
     */
    const std::size_t get_num_sites() const; 

    /**
     * @brief returns the number of sequences this profile was generated for
     */
    const std::size_t get_num_profiles() const; 


    /**
     * @brief returns the number of rows, should be either 5 or more for proteins
     */
    const std::size_t get_num_rows() const; 


    /**
     * @brief fastly swaps *this, and the parameter profile
     */
    void swap( Profile & rhs );
  private:
    void inner_create_profiles( AlignCode const& source, tree_types::profile_map const& tree );
    void inner_create_profiles( AlignCode const& source, std::list<std::list<std::size_t> > const& profiles );

    void process_difference( SingleProfile & profile, AlignCode const& code, std::size_t index );

    template<typename IteratorT>
    void handle_differences( SingleProfile & profile, AlignCode const& code, IteratorT begin, IteratorT end )
    {
      for(;begin!=end; ++begin ) 
        if( *begin )
          process_difference( profile, code, *begin - 1 );
    }



    std::size_t num_sites, num_profiles, num_rows;
    con_type profile_array; 
};


/**
 * @brief fastly swaps the two profile
 */
void swap( Profile &lhs, Profile & rhs );

/**
 * @brief displays debug info onto console
 */
std::ostream & operator<<( std::ostream & o, Profile const& obj );

#endif


