/***************************************************************************
 *   Copyright (C) 2005 by Andreas Pokorny                                 *
 *   andreas.pokorny@biozentrum.uni-wuerzburg.de                           *
 *                                                                         *
 *   This file is part of profdist and cbcanalyzer                         *
 *                                                                         *
 *   Both profdist and cbcanalyzer are free software; you can redistribute * 
 *   it and/or modify it under the terms of the GNU General Public License * 
 *   as published by the Free Software Foundation; either version 2 of the * 
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 *   Profdist and cbcanalyzer are distributed in the hope that it will be  *
 *   useful, but WITHOUT ANY WARRANTY; without even the implied warranty   *
 *   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the      *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/**
 * @file parser.h is a file containing all parse routines, for dna, and tree formats
 * TODO turn all that into templates .. and have everything inline. Either there is no 
 * need for more than two different iterator types. Or there is a need for an iterator 
 * different to position<file_iterator<..> >
 * @author Andreas Pokorny
 * @data 14.12.2004 
 */

#ifndef PROFDIST_PARSER_H_D 
#define PROFDIST_PARSER_H_D 1

#include <vector>
#include <string>
#include <boost/spirit/iterator/file_iterator.hpp>
#include <boost/spirit/iterator/position_iterator.hpp>

#include "parsed_sequence.h"
#include "parsed_tree.h"


typedef sequence<boost::spirit::position_iterator<boost::spirit::file_iterator<char > > > file_sequence;

/**
 * @brief Parses a Fasta file, and write all found sequences into the list parameter. 
 * The function will throw on error, even then all successfully parsed sequences
 * can be found in the list. Catch runtime_error when using this function.
 * @param[in] filename path to the file to parse
 * @param[out] sequences the results of the parse operations, will not be cleared before parsing
 */
void parse_fasta( std::string const& filename, std::list<file_sequence>& sequences );


/**
 * @brief Parses the output of the embl database, and write all found sequences into the 
 * list parameter. The function will throw on error, even then all successfully 
 * parsed sequences can be found in the list. The classification information will be 
 * extracted too. Catch runtime_error when using this function.
 * @param[in] filename path to the file to parse
 * @param[out] sequences the results of the parse operations, will not be cleared before parsing
 */
void parse_embl( std::string const& filename, std::list<file_sequence>& sequences );

/**
 * @brief Parses ct files (ct, vienna ct, and mac ct, and more? ).
 * The function will throw on error, even then all successfully parsed sequences
 * can be found in the list. Catch runtime_error when using this function.
 * @param[in] filename path to the file to parse
 * @param[out] sequences the results of the parse operations, will not be cleared before parsing
 */
void parse_ct( std::string const& filename, std::list<file_sequence>& sequences );

/**
 * @brief Parses dumps of RNAforester.
 * The function will throw on error, even then all successfully parsed sequences
 * can be found in the list. Catch runtime_error when using this function.
 * @param[in] filename path to the file to parse
 * @param[out] sequences the results of the parse operations, will not be cleared before parsing
 */
void parse_forester( std::string const& filename, std::list<file_sequence>& sequences );

/**
 * Paser bracket-dot-bracket file format
 * The function will throw on error, even then all successfully parsed sequences
 * can be found in the list. Catch runtime_error when using this function.
 * @param[in] filename path to the file to parse
 * @param[out] sequences the results of the parse operations, will not be cleared before parsing
 */
void parse_bdb( std::string const& filename, std::list<file_sequence>& sequences );

/**
 * Parse dump of Marna.
 * The function will throw on error, even then all successfully parsed sequences
 * can be found in the list. Catch runtime_error when using this function.
 * @param[in] filename path to the file to parse
 * @param[out] sequences the results of the parse operations, will not be cleared before parsing
 */
void parse_marna( std::string const& filename, std::list<file_sequence>& sequences );


/**
 * Parse newick tree format.
 * The function will throw on error, even then all successfully parsed sequences
 * can be found in the list. Catch runtime_error when using this function.
 * @param[in] filename path to the file to parse
 * @param[out] trees contains the result of the partial or complete parsing operation
 * @param[out] sequence_names container with all names found in the file
 */
void parse_newick( std::string const& filename, std::list<profdist::parsed_node::node_ptr>& trees, std::vector<std::string> & sequence_names );

#endif 
  
